package org.flywaydb.core.internal.database.openguass;

import com.google.auto.service.AutoService;
import lombok.extern.slf4j.Slf4j;
import org.flywaydb.core.api.ResourceProvider;
import org.flywaydb.core.api.configuration.Configuration;
import org.flywaydb.core.extensibility.Plugin;
import org.flywaydb.core.internal.authentication.postgres.PgpassFileReader;
import org.flywaydb.core.internal.database.base.BaseDatabaseType;
import org.flywaydb.core.internal.database.base.Database;
import org.flywaydb.core.internal.database.kingbase.KingbaseDatabase;
import org.flywaydb.core.internal.database.kingbase.KingbaseParser;
import org.flywaydb.core.internal.database.postgresql.PostgreSQLDatabase;
import org.flywaydb.core.internal.database.postgresql.PostgreSQLParser;
import org.flywaydb.core.internal.jdbc.JdbcConnectionFactory;
import org.flywaydb.core.internal.jdbc.StatementInterceptor;
import org.flywaydb.core.internal.parser.Parser;
import org.flywaydb.core.internal.parser.ParsingContext;

import java.sql.Connection;
import java.sql.Types;
import java.util.Properties;

/**
 * @author xu.wenchang
 * @version 1.0 2023/10/17
 */
@AutoService(Plugin.class)
@Slf4j
public class OpenGuassDatabaseType extends BaseDatabaseType {
    @Override
    public String getName() {
        return "PostgreSQL";
    }
    
    @Override
    public int getNullType() {
        return Types.NULL;
    }
    
    @Override
    public boolean handlesJDBCUrl(String url) {
        return url.startsWith("jdbc:opengauss:");
    }
    
    @Override
    public String getDriverClass(String url, ClassLoader classLoader) {
        return "org.opengauss.Driver";
    }
    
    @Override
    public boolean handlesDatabaseProductNameAndVersion(String databaseProductName, String databaseProductVersion,
                                                        Connection connection) {
        return databaseProductName.startsWith("PostgreSQL");
    }
    
    @Override
    public Database createDatabase(Configuration configuration, JdbcConnectionFactory jdbcConnectionFactory,
                                   StatementInterceptor statementInterceptor) {
        return new PostgreSQLDatabase(configuration, jdbcConnectionFactory, statementInterceptor);
    }
    
    @Override
    public Parser createParser(Configuration configuration, ResourceProvider resourceProvider,
                               ParsingContext parsingContext) {
        return new PostgreSQLParser(configuration, parsingContext);
    }
    
    @Override
    public void setDefaultConnectionProps(String url, Properties props, ClassLoader classLoader) {
        props.put("applicationName", APPLICATION_NAME);
    }
    
    @Override
    public boolean detectUserRequiredByUrl(String url) {
        return !url.contains("user=");
    }
    
    @Override
    public boolean detectPasswordRequiredByUrl(String url) {
        // Postgres supports password in URL
        return !url.contains("password=");
    }
    
    @Override
    public boolean externalAuthPropertiesRequired(String url, String username, String password) {
        return super.externalAuthPropertiesRequired(url, username, password);
    }
    
    @Override
    public Properties getExternalAuthProperties(String url, String username) {
        PgpassFileReader pgpassFileReader = new PgpassFileReader();
        
        if (pgpassFileReader.getPgpassFilePath() != null) {
            log.info(org.flywaydb.core.internal.license.FlywayTeamsUpgradeMessage.generate("pgpass file '" + pgpassFileReader.getPgpassFilePath() + "'", "use this for database authentication"));
        }
        
        return super.getExternalAuthProperties(url, username);
    }
}