package org.violet.common.core.base;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.servlet.ModelAndView;
import org.violet.common.core.util.MimeUtil;


import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public class BaseController {
    @Autowired
    public HttpServletRequest request;
    @Autowired
    public HttpServletResponse response;

    public BaseController() {
    }

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        binder.setAutoGrowCollectionLimit(10000);
    }

    public HttpHeaders buildDownloadHeader(String fileName) {
        HttpHeaders headers = new HttpHeaders();

        try {
            headers.add("Cache-Control", "no-cache, no-store, must-revalidate");
            headers.add("Content-Disposition", "attachment; filename=\"" + URLEncoder.encode(fileName, StandardCharsets.UTF_8.name()) + "\"");
            headers.add("Pragma", "no-cache");
            headers.add("Expires", "0");
            return headers;
        } catch (Exception var4) {
            throw new RuntimeException("headers构建失败，原因是：" + var4.getMessage(), var4);
        }
    }

    public ResponseEntity<FileSystemResource> renderFile(File file) {
        return file == null ? null : (ResponseEntity.ok().headers(this.buildDownloadHeader(file.getName()))).contentLength(file.length()).contentType(MediaType.parseMediaType("application/octet-stream")).body(new FileSystemResource(file));
    }

    public ResponseEntity<InputStreamResource> renderFile(InputStream inputStream, String fileName, long size) {
        return inputStream == null ? null : (ResponseEntity.ok().headers(this.buildDownloadHeader(fileName))).contentLength(size).contentType(MediaType.parseMediaType("application/octet-stream")).body(new InputStreamResource(inputStream));
    }

    public ResponseEntity<FileSystemResource> renderFile(File file, String fileType) {
        return file == null ? null : (ResponseEntity.ok().headers(this.buildDownloadHeader(file.getName()))).contentLength(file.length()).contentType(MediaType.parseMediaType(fileType)).body(new FileSystemResource(file));
    }

    public ResponseEntity<ByteArrayResource> renderFile(byte[] bytes, String fileType, String fileName) {
        return bytes.length < 1 ? null : (ResponseEntity.ok().headers(this.buildDownloadHeader(fileName))).contentLength((long) bytes.length).contentType(StrUtil.isNotEmpty(fileType) ? MediaType.parseMediaType(fileType) : MediaType.parseMediaType("application/octet-stream")).body(new ByteArrayResource(bytes));
    }

    public ResponseEntity<ByteArrayResource> renderFile(byte[] bytes, String fileName) {
        return bytes.length < 1 ? null : (ResponseEntity.ok().headers(this.buildDownloadHeader(fileName))).contentLength((long) bytes.length).contentType(MediaType.parseMediaType("application/octet-stream")).body(new ByteArrayResource(bytes));
    }

    public void renderImage(byte[] bytes, String imageName) {
        String contentType = MimeUtil.getMimeType(imageName);
        this.response.setContentType(contentType);
        ServletOutputStream os = null;

        try {
            os = this.response.getOutputStream();
            os.write(bytes);
            os.flush();
            os.close();
        } catch (IOException var6) {
            var6.printStackTrace();
        }

    }

    public ModelAndView view(String viewname) {
        return this.view((String) viewname, (Map) null);
    }

    public ModelAndView view(String viewname, Map<String, Object> modelMap) {
        ModelAndView view = new ModelAndView();
        view.setViewName(viewname);
        view.addAllObjects(modelMap);
        return view;
    }

    public ModelAndView view(ModelAndView view, Map<String, Object> modelMap) {
        view.addAllObjects(modelMap);
        return view;
    }

    public Map<String, Object> getParamsMap() {
        Map<String, Object> params = new HashMap();
        Map<String, String[]> map = this.request.getParameterMap();
        Iterator var3 = map.keySet().iterator();

        while (var3.hasNext()) {
            String key = (String) var3.next();
            String[] values = (String[]) map.get(key);
            params.put(key, ArrayUtil.join(values, ","));
        }

        return params;
    }
}