package org.violet.common.core.util;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.fastjson2.JSON;

import lombok.Getter;
import lombok.Setter;
import org.violet.common.core.exception.BizException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 级联工具类
 *
 * @author xu.wenchang
 * @version 1.0 2023/07/31
 */
public class CascadeUtil {
    @Getter
    @Setter
    public static class CascadeItem {
        private final Object key;
        private final Object parentKey;
        private final String text;
        private String cascadeText = "";
        private List<CascadeItem> children;
        
        public CascadeItem(Object key, Object parentKey, String text) {
            this.key = key;
            this.parentKey = parentKey;
            this.text = text;
        }
    }
    
    public static void main(String[] args) {
        List<CascadeItem> data = new ArrayList<>();
        CascadeItem a = new CascadeItem(1, 0, "a");
        data.add(a);
    
        CascadeItem a1 = new CascadeItem(2, 1, "a1");
        data.add(a1);
        
        CascadeItem a2 = new CascadeItem(3, 1, "a2");
        data.add(a2);
    
        CascadeItem a11 = new CascadeItem(4, 2, "a11");
        data.add(a11);
    
        CascadeItem b = new CascadeItem(5, 0, "b");
        data.add(b);
        cascade(data,"-");
        System.out.println(JSON.toJSONString(data));
    }
    
    /**
     * 构造级联数据
     *
     * @param data      原始数据
     * @param connector 连接符，连接每级文字
     */
    public static void cascade(List<CascadeItem> data, String connector) {
        if (CollUtil.isEmpty(data)) {
            return;
        }
        
        Map<Object, CascadeItem> map = new HashMap<>();
        for (CascadeItem item : data) {
            map.put(item.getKey(), item);
        }
        
        try {
            for (CascadeItem item : data) {
                Object keyVal = item.getKey();
                Object parentVal = item.getParentKey();
                map.put(keyVal, item);
                
                if (!isRootNode(parentVal)) {
                    CascadeItem parent = map.get(parentVal);
                    
                    if (null != parent) {
                        if (null == parent.getChildren()) {
                            parent.setChildren(new ArrayList<>());
                        }
                        parent.getChildren()
                              .add(item);
                    }
                }
            }
            
            List<CascadeItem> pool = map.values()
                                        .stream()
                                        .filter(row -> isRootNode(row.getParentKey()))
                                        .collect(Collectors.toList());
            
            while (pool.size() > 0) {
                CascadeItem current = pool.remove(0);
                Object parentVal = current.getParentKey();
                String cascadeVal;
                
                if (isRootNode(parentVal)) {
                    cascadeVal = current.getText();
                } else {
                    cascadeVal = current.getCascadeText() + connector + current.getText();
                }
                
                current.setCascadeText(cascadeVal);
                List<CascadeItem> children = current.getChildren();
                current.setChildren(null);
                if (CollUtil.isNotEmpty(children)) {
                    for (CascadeItem item : children) {
                        item.setCascadeText(cascadeVal);
                    }
                    pool.addAll(children);
                }
            }
        } catch (Exception e) {
            throw new BizException(e.getMessage(), e);
        }
    }
    
    private static boolean isRootNode(Object parentValue) {
        if (null == parentValue) {
            return true;
        }
        
        if (parentValue instanceof String) {
            return ((String) parentValue).trim()
                                         .equals("0");
        }
        
        if (parentValue instanceof Integer) {
            return (Integer) parentValue == 0;
        }
        
        if (parentValue instanceof Long) {
            return (Long) parentValue == 0;
        }
        
        return false;
    }
}
