package org.violet.common.cloud.feign;

import com.fasterxml.jackson.databind.JsonNode;
import org.violet.common.launch.entity.JsonResult;
import feign.FeignException;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.springframework.cglib.proxy.MethodInterceptor;
import org.springframework.cglib.proxy.MethodProxy;
import org.springframework.lang.Nullable;
import org.violet.common.core.util.JsonUtil;
import org.violet.common.core.util.ObjectUtil;

import java.lang.reflect.Method;
import java.util.Objects;

/**
 * fallBack 代理处理，降级处理，用于feign和sentinel整合
 */
@Slf4j
@AllArgsConstructor
public class VioletFeignFallback<T> implements MethodInterceptor {
    private final Class<T> targetType;
    private final String targetName;
    private final Throwable cause;
    private final static String CODE = "code";

    //服务被限流降级默认返回
    @Nullable
    @Override
    public Object intercept(Object o, Method method, Object[] objects, MethodProxy methodProxy) throws Throwable {
        //获取异常信息
        String errorMessage = cause.getMessage();
        log.error("VioletFeignFallback:[{}.{}] serviceId:[{}] message:[{}]", targetType.getName(), method.getName(), targetName, errorMessage);
        //获取返回类型
        Class<?> returnType = method.getReturnType();
        // 暂时不支持 flux，rx，异步等，返回值不是 JsonResult，直接返回 null。
        if (JsonResult.class != returnType) {
            return null;
        }
        // 非 FeignException
        if (!(cause instanceof FeignException)) {
            return JsonResult.ERROR(errorMessage);
        }
        FeignException exception = (FeignException) cause;
        byte[] content = exception.content();
        // 如果返回的数据为空
        if (ObjectUtil.isEmpty(content)) {
            return JsonResult.ERROR(errorMessage);
        }
        // 转换成 jsonNode 读取，因为直接转换，可能 对方放回的并 不是 jsonresult 的格式。

        try {
            JsonNode resultNode = null;
            resultNode = JsonUtil.readTree(content);
            // 判断是否 JsonResult 格式 返回体
            if (resultNode.has(CODE)) {
                return JsonUtil.getInstance().convertValue(resultNode, JsonResult.class);
            } else {
                return JsonResult.ERROR(resultNode.toString());
            }
        } catch (Exception ex) {

        }
        return JsonResult.ERROR(errorMessage);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        VioletFeignFallback<?> that = (VioletFeignFallback<?>) o;
        return targetType.equals(that.targetType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(targetType);
    }
}
