/*
 * Decompiled with CFR 0.152.
 */
package shaded.cfca.sadk.org.bouncycastle.cert.ocsp.test;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.Security;
import java.util.Date;
import java.util.Random;
import java.util.Set;
import java.util.Vector;
import shaded.cfca.sadk.org.bouncycastle.asn1.ASN1Encodable;
import shaded.cfca.sadk.org.bouncycastle.asn1.ASN1Exception;
import shaded.cfca.sadk.org.bouncycastle.asn1.ASN1OctetString;
import shaded.cfca.sadk.org.bouncycastle.asn1.DEROctetString;
import shaded.cfca.sadk.org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import shaded.cfca.sadk.org.bouncycastle.asn1.x509.Extension;
import shaded.cfca.sadk.org.bouncycastle.asn1.x509.ExtensionsGenerator;
import shaded.cfca.sadk.org.bouncycastle.asn1.x509.GeneralName;
import shaded.cfca.sadk.org.bouncycastle.asn1.x509.X509Name;
import shaded.cfca.sadk.org.bouncycastle.cert.CertIOException;
import shaded.cfca.sadk.org.bouncycastle.cert.X509CertificateHolder;
import shaded.cfca.sadk.org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.BasicOCSPResp;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.CertificateID;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.CertificateStatus;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.OCSPReq;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.OCSPReqBuilder;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.OCSPResp;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.OCSPRespBuilder;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.Req;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.RespID;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.SingleResp;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.jcajce.JcaBasicOCSPRespBuilder;
import shaded.cfca.sadk.org.bouncycastle.cert.ocsp.test.OCSPTestUtil;
import shaded.cfca.sadk.org.bouncycastle.jce.X509Principal;
import shaded.cfca.sadk.org.bouncycastle.jce.provider.BouncyCastleProvider;
import shaded.cfca.sadk.org.bouncycastle.operator.DigestCalculatorProvider;
import shaded.cfca.sadk.org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import shaded.cfca.sadk.org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import shaded.cfca.sadk.org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import shaded.cfca.sadk.org.bouncycastle.util.encoders.Base64;
import shaded.cfca.sadk.org.bouncycastle.util.test.SimpleTest;

public class OCSPTest
extends SimpleTest {
    byte[] testResp1 = Base64.decode("MIIFnAoBAKCCBZUwggWRBgkrBgEFBQcwAQEEggWCMIIFfjCCARehgZ8wgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4YDzIwMDMwNDAyMTIzNDU4WjBiMGAwOjAJBgUrDgMCGgUABBRs07IuoCWNmcEl1oHwIak1BPnX8QQUtGyl/iL9WJ1VxjxFj0hAwJ/s1AcCAQKhERgPMjAwMjA4MjkwNzA5MjZaGA8yMDAzMDQwMjEyMzQ1OFowDQYJKoZIhvcNAQEFBQADgYEAfbN0TCRFKdhsmvOdUoiJ+qvygGBzDxD/VWhXYA+16AphHLIWNABR3CgHB3zWtdy2j7DJmQ/R7qKj7dUhWLSqclAiPgFtQQ1YvSJAYfEIdyHkxv4NP0LSogxrumANcDyC9yt/W9yHjD2ICPBIqCsZLuLkOHYi5DlwWe9Zm9VFwCGgggPMMIIDyDCCA8QwggKsoAMCAQICAQYwDQYJKoZIhvcNAQEFBQAwgZQxFDASBgNVBAMTC1RDUy1DQSBPQ1NQMSYwJAYJKoZIhvcNAQkBFhd0Y3MtY2FAdGNzLWNhLnRjcy5jby5pbjEMMAoGA1UEChMDVENTMQwwCgYDVQQLEwNBVEMxEjAQBgNVBAcTCUh5ZGVyYWJhZDEXMBUGA1UECBMOQW5kaHJhIHByYWRlc2gxCzAJBgNVBAYTAklOMB4XDTAyMDgyOTA3MTE0M1oXDTAzMDgyOTA3MTE0M1owgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAM+XWW4caMRv46D7L6Bv8iwtKgmQu0SAybmFRJiz12qXzdvTLt8C75OdgmUomxp0+gW/4XlTPUqOMQWv463aZRv9Ust4f8MHEJh4ekP/NS9+d8vEO3P40ntQkmSMcFmtA9E1koUtQ3MSJlcs441JjbgUaVnmjDmmniQnZY4bU3tVAgMBAAGjgZowgZcwDAYDVR0TAQH/BAIwADALBgNVHQ8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwkwNgYIKwYBBQUHAQEEKjAoMCYGCCsGAQUFBzABhhpodHRwOi8vMTcyLjE5LjQwLjExMDo3NzAwLzAtBgNVHR8EJjAkMCKgIKAehhxodHRwOi8vMTcyLjE5LjQwLjExMC9jcmwuY3JsMA0GCSqGSIb3DQEBBQUAA4IBAQB6FovM3B4VDDZ15o12gnADZsIk9fTAczLlcrmXLNN4PgmqgnwF0Ymj3bD5SavDOXxbA65AZJ7rBNAguLUo+xVkgxmoBH7R2sBxjTCcr07NEadxM3HQkt0aX5XYEl8eRoifwqYAI9h0ziZfTNes8elNfb3DoPPjqq6VmMg0f0iMS4W8LjNPorjRB+kIosa1deAGPhq0eJ8yr0/s2QR2/WFD5P4aXc8IKWleklnIImS3zqiPrq6tl2Bm8DZj7vXlTOwmraSQxUwzCKwYob1yGvNOUQTqpG6jxn7jgDawHU1+WjWQe4Q34/pWeGLysxTraMa+Ug9kPe+jy/qRX2xwvKBZ");
    byte[] testResp2 = Base64.decode("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");
    private static byte[] irregReq = Base64.decode("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");
    private static byte[] invalidResp = Base64.decode("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");
    private static final String BC = "BC";

    public String getName() {
        return "OCSP";
    }

    private void testECDSA() throws Exception {
        Extension extValue;
        ASN1Encodable extObj;
        Set extOids;
        byte[] reqEnc;
        OCSPReq newReq;
        Req[] requests;
        X509CertificateHolder[] certs;
        String signDN = "O=Bouncy Castle, C=AU";
        KeyPair signKP = OCSPTestUtil.makeECKeyPair();
        JcaX509CertificateHolder testCert = new JcaX509CertificateHolder(OCSPTestUtil.makeECDSACertificate(signKP, signDN, signKP, signDN));
        DigestCalculatorProvider digCalcProv = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
        String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName origName = new GeneralName(new X509Name(origDN));
        CertificateID id = new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L));
        OCSPReqBuilder gen = new OCSPReqBuilder();
        gen.addRequest(id);
        OCSPReq req = gen.build();
        if (req.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((certs = req.getCerts()).length != 0) {
            this.fail("0 certs expected, but not found");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        X509CertificateHolder[] chain = new X509CertificateHolder[1];
        gen = new OCSPReqBuilder();
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(signKP.getPrivate()), chain);
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((certs = req.getCerts()) == null) {
            this.fail("null certs found");
        }
        if (certs.length != 1 || !certs[0].equals(testCert)) {
            this.fail("incorrect certs found in request");
        }
        if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("newReq signature failed to verify");
        }
        chain = new X509CertificateHolder[1];
        gen = new OCSPReqBuilder();
        Vector oids = new Vector();
        Vector values = new Vector();
        byte[] sampleNonce = new byte[16];
        Random rand = new Random();
        rand.nextBytes(sampleNonce);
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        ExtensionsGenerator extGen = new ExtensionsGenerator();
        extGen.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, new DEROctetString(sampleNonce));
        gen.setRequestExtensions(extGen.generate());
        gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(signKP.getPrivate()), chain);
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((extObj = (extValue = req.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
            this.fail("wrong extension value found.");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        JcaBasicOCSPRespBuilder respGen = new JcaBasicOCSPRespBuilder(signKP.getPublic(), digCalcProv.get(RespID.HASH_SHA1));
        respGen.addResponse(id, CertificateStatus.GOOD);
        BasicOCSPResp resp = respGen.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(signKP.getPrivate()), chain, new Date());
    }

    private void testRSA() throws Exception {
        Extension ext;
        ASN1Encodable extObj;
        Set extOids;
        byte[] reqEnc;
        OCSPReq newReq;
        Req[] requests;
        X509CertificateHolder[] certs;
        String signDN = "O=Bouncy Castle, C=AU";
        KeyPair signKP = OCSPTestUtil.makeKeyPair();
        JcaX509CertificateHolder testCert = new JcaX509CertificateHolder(OCSPTestUtil.makeCertificate(signKP, signDN, signKP, signDN));
        DigestCalculatorProvider digCalcProv = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
        String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName origName = new GeneralName(new X509Name(origDN));
        CertificateID id = new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L));
        OCSPReqBuilder gen = new OCSPReqBuilder();
        gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
        OCSPReq req = gen.build();
        if (req.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((certs = req.getCerts()).length != 0) {
            this.fail("0 certs expected, but not found");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        X509CertificateHolder[] chain = new X509CertificateHolder[1];
        gen = new OCSPReqBuilder();
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(signKP.getPrivate()), chain);
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((certs = req.getCerts()) == null) {
            this.fail("null certs found");
        }
        if (certs.length != 1 || !certs[0].equals(testCert)) {
            this.fail("incorrect certs found in request");
        }
        if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("newReq signature failed to verify");
        }
        chain = new X509CertificateHolder[1];
        gen = new OCSPReqBuilder();
        byte[] sampleNonce = new byte[16];
        Random rand = new Random();
        rand.nextBytes(sampleNonce);
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        ExtensionsGenerator extGen = new ExtensionsGenerator();
        extGen.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, new DEROctetString(sampleNonce));
        gen.setRequestExtensions(extGen.generate());
        gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(signKP.getPrivate()), chain);
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((extObj = (ext = req.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
            this.fail("wrong extension value found.");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        JcaBasicOCSPRespBuilder respGen = new JcaBasicOCSPRespBuilder(signKP.getPublic(), digCalcProv.get(RespID.HASH_SHA1));
        respGen.addResponse(id, CertificateStatus.GOOD);
        BasicOCSPResp resp = respGen.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(signKP.getPrivate()), chain, new Date());
        OCSPRespBuilder rGen = new OCSPRespBuilder();
        byte[] enc = rGen.build(0, resp).getEncoded();
    }

    private void testIrregularVersionReq() throws Exception {
        OCSPReq ocspRequest = new OCSPReq(irregReq);
        X509CertificateHolder cert = ocspRequest.getCerts()[0];
        if (!ocspRequest.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(cert))) {
            this.fail("extra version encoding test failed");
        }
    }

    public void testInvalidResp() throws Exception {
        try {
            OCSPResp response = new OCSPResp(invalidResp);
        }
        catch (CertIOException e) {
            if (e.getCause() instanceof ASN1Exception) {
                Throwable c = ((ASN1Exception)e.getCause()).getCause();
                if (!c.getMessage().equals("ENUMERATED has zero length")) {
                    this.fail("parsing failed, but for wrong reason: " + c.getMessage());
                }
            }
            this.fail("parsing failed, but for wrong reason: " + e.getMessage());
        }
    }

    public void performTest() throws Exception {
        block26: {
            OCSPReq req;
            block25: {
                OCSPResp response;
                Extension ext;
                ASN1Encodable extObj;
                Set extOids;
                byte[] reqEnc;
                OCSPReq newReq;
                Req[] requests;
                X509CertificateHolder[] certs;
                String signDN = "O=Bouncy Castle, C=AU";
                KeyPair signKP = OCSPTestUtil.makeKeyPair();
                JcaX509CertificateHolder testCert = new JcaX509CertificateHolder(OCSPTestUtil.makeCertificate(signKP, signDN, signKP, signDN));
                String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
                GeneralName origName = new GeneralName(new X509Name(origDN));
                DigestCalculatorProvider digCalcProv = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
                CertificateID id = new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L));
                OCSPReqBuilder gen = new OCSPReqBuilder();
                gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
                req = gen.build();
                if (req.isSigned()) {
                    this.fail("signed but shouldn't be");
                }
                if ((certs = req.getCerts()).length != 0) {
                    this.fail("0 certs expected, but not found");
                }
                if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
                    this.fail("Failed isFor test");
                }
                X509CertificateHolder[] chain = new X509CertificateHolder[1];
                gen = new OCSPReqBuilder();
                gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
                gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
                chain[0] = testCert;
                req = gen.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(signKP.getPrivate()), chain);
                if (!req.isSigned()) {
                    this.fail("not signed but should be");
                }
                if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
                    this.fail("signature failed to verify");
                }
                if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
                    this.fail("Failed isFor test");
                }
                if ((certs = req.getCerts()) == null) {
                    this.fail("null certs found");
                }
                if (certs.length != 1 || !certs[0].equals(testCert)) {
                    this.fail("incorrect certs found in request");
                }
                if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
                    this.fail("newReq signature failed to verify");
                }
                chain = new X509CertificateHolder[1];
                gen = new OCSPReqBuilder();
                Vector oids = new Vector();
                Vector values = new Vector();
                byte[] sampleNonce = new byte[16];
                Random rand = new Random();
                rand.nextBytes(sampleNonce);
                gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
                ExtensionsGenerator extGen = new ExtensionsGenerator();
                extGen.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, new DEROctetString(sampleNonce));
                gen.setRequestExtensions(extGen.generate());
                gen.addRequest(new CertificateID(digCalcProv.get(CertificateID.HASH_SHA1), testCert, BigInteger.valueOf(1L)));
                chain[0] = testCert;
                req = gen.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(signKP.getPrivate()), chain);
                if (!req.isSigned()) {
                    this.fail("not signed but should be");
                }
                if (!req.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(signKP.getPublic()))) {
                    this.fail("signature failed to verify");
                }
                if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
                    this.fail("wrong number of critical extensions in OCSP request.");
                }
                if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
                    this.fail("wrong number of non-critical extensions in OCSP request.");
                }
                if (!((extObj = (ext = req.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
                    this.fail("wrong extension type found.");
                }
                if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
                    this.fail("wrong extension value found.");
                }
                if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
                    this.fail("Failed isFor test");
                }
                if ((response = new OCSPResp(this.testResp1)).getStatus() != 0) {
                    this.fail("response status not zero.");
                }
                BasicOCSPResp brep = (BasicOCSPResp)response.getResponseObject();
                chain = brep.getCerts();
                if (!brep.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(chain[0]))) {
                    this.fail("response 1 failed to verify.");
                }
                SingleResp[] singleResp = brep.getResponses();
                response = new OCSPResp(this.testResp2);
                if (response.getStatus() != 0) {
                    this.fail("response status not zero.");
                }
                brep = (BasicOCSPResp)response.getResponseObject();
                chain = brep.getCerts();
                if (!brep.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(chain[0]))) {
                    this.fail("response 2 failed to verify.");
                }
                singleResp = brep.getResponses();
                OCSPRespBuilder respGen = new OCSPRespBuilder();
                OCSPResp resp = respGen.build(0, response.getResponseObject());
                if (!resp.getResponseObject().equals(response.getResponseObject())) {
                    this.fail("response fails to match");
                }
                this.testECDSA();
                this.testRSA();
                this.testIrregularVersionReq();
                this.testInvalidResp();
                try {
                    response = new OCSPResp(new byte[0]);
                    this.fail("no exception thrown");
                }
                catch (IOException e) {
                    if (e.getMessage().equals("malformed response: no response data found")) break block25;
                    this.fail("wrong exception");
                }
            }
            try {
                req = new OCSPReq(new byte[0]);
                this.fail("no exception thrown");
            }
            catch (IOException e) {
                if (e.getMessage().equals("malformed request: no request data found")) break block26;
                this.fail("wrong exception");
            }
        }
    }

    public static void main(String[] args) {
        Security.addProvider(new BouncyCastleProvider());
        OCSPTest.runTest(new OCSPTest());
    }
}

